/***************************************************************************
 *   Copyright (C) 2002-2019 by Victor Julien                              *
 *   victor@vuurmuur.org                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "textdir_plugin.h"

/*
    asking from and telling to the backend (TODO: name)

    returns
        -1 error
*/
int ask_textdir(void *backend, const char *name, const char *question,
        char *answer, size_t max_answer, enum vrmr_objecttypes type, int multi)
{
    int retval = 0;
    char *file_location = NULL;
    char line[MAX_LINE_LENGTH] = "", variable[64] = "", value[512] = "";
    size_t line_pos = 0, val_pos = 0;
    size_t line_length = 0;
    size_t len = 0;

    assert(backend && name && question);

    vrmr_debug(
            HIGH, "question: %s, name: %s, multi: %d", question, name, multi);

    struct textdir_backend *tb = (struct textdir_backend *)backend;
    if (!tb->backend_open) {
        vrmr_error(-1, "Error", "backend not opened yet");
        return (-1);
    }

    /* determine the location of the file */
    if (!(file_location = get_filelocation(backend, name, type)))
        return (-1);

    /* check if we are clean */
    if (tb->file != NULL && multi == 0) {
        vrmr_warning("Warning",
                "the last 'multi' call to '%s' probably failed, because the "
                "file is still open when it shouldn't",
                name);

        fclose(tb->file);
        tb->file = NULL;
    }

    /* now open and read the file, but only if it is not already open */
    if (tb->file == NULL) {
        if (!(tb->file = vuurmuur_fopen(tb->cfg, file_location, "r"))) {
            vrmr_error(-1, "Error", "Unable to open file '%s'.", file_location);

            free(file_location);
            return (-1);
        }
    }

    /* start (or continue) looping trough the file */
    while (fgets(line, (int)sizeof(line), tb->file) != NULL) {
        line_length = strlen(line);
        if (line_length > MAX_LINE_LENGTH) {
            vrmr_error(-1, "Error",
                    "line is longer than allowed (line: %d, max: %d)",
                    (int)line_length, MAX_LINE_LENGTH);

            free(file_location);
            fclose(tb->file);
            tb->file = NULL;
            return (-1);
        }

        /* first check if the line is a comment. */
        if (line_length == 0 || line[0] == '#' || line[0] == ' ' ||
                line[0] == '\0' || line[0] == '\n' || line[0] == '\t') {
            /* continue with the next line, its a comment or an empty line. */
            continue;
        }

        /* look for the occurance of the = separator */
        char *val = strchr(line, '=');
        if (val == NULL) {
            /* not a valid line, ignore */
            continue;
        }

        /* val - line = var len */
        size_t var_len = val - line + 1;
        if (var_len > (sizeof(variable) - 1)) {
            /* invalid line, ignore */
            continue;
        }
        strlcpy(variable, line, var_len);
        vrmr_debug(LOW, "variable %s", variable);

        /* now see if this was what we were looking for */
        if (strcasecmp(question, variable) != 0) {
            /* nope, ignore line */
            continue;
        }

        /* skip pass the '=' char */
        val++;

        size_t val_len = strlen(val);

        /* copy the value into "value" */
        val_pos = 0;
        line_pos = 0;

        while (val[line_pos] != '\0' && val[line_pos] != '\n' &&
                line_pos < val_len && val_pos < max_answer) {
            /* if the first character is a '"' we strip it. */
            if ((val_pos == 0) && (val[line_pos] == '\"'))
                line_pos++;

            /* otherwise copy the char */
            else {
                value[val_pos] = val[line_pos];

                line_pos++;
                val_pos++;
            }
        }

        /* if the last character is a '"' we strip it. */
        if (val_pos > 0 && value[val_pos - 1] == '\"')
            value[val_pos - 1] = '\0';
        else
            value[val_pos] = '\0';

        vrmr_debug(
                MEDIUM, "question '%s' matched, value: '%s'", question, value);

        /* copy back the value to "answer" */
        len = strlcpy(answer, value, max_answer);
        if (len >= max_answer) {
            vrmr_error(-1, "Error",
                    "buffer overrun when reading file '%s', question '%s': len "
                    "%u, max: %u",
                    file_location, question, (int)len, (int)max_answer);

            free(file_location);
            fclose(tb->file);
            tb->file = NULL;
            return (-1);
        }

        /* only return when bigger than 0 */
        if (strlen(answer) > 0)
            retval = 1;

        /* break out of the loop so when we call multi again we continue where
         * we were */
        break;
    }

    /* cleanup */
    if ((multi == 1 && retval != 1) || multi == 0) {
        vrmr_debug(HIGH, "close the file.");

        if (fclose(tb->file) != 0) {
            vrmr_error(-1, "Error", "closing file '%s' failed: %s",
                    file_location, strerror(errno));
            retval = -1;
        }
        tb->file = NULL;
    }

    /* cleanup filelocation */
    free(file_location);

    vrmr_debug(HIGH, "at exit: tb->file: %p (retval: %d).", tb->file, retval);
    vrmr_debug(HIGH, "** end **, retval=%d", retval);

    return (retval);
}
