/***************************************************************************
 *   Copyright (C) 2002-2006 by Victor Julien                              *
 *   victor@nk.nl                                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/*************************************************************************** 
 * Here we try to analize the actual rule.                                  *
 ***************************************************************************/
#include "vuurmuur.h"


/*	rules_analyse_rule

	Function for gathering the info for creation of the rule
	and for sanity checking the rule.

	Returncodes:
		 0: ok
		-1: error
 */
int
rules_analyze_rule(	const int debuglvl,
			struct RuleData_ *rule_ptr,
			struct RuleCache_ *create,
			Services *services,
			Zones *zones,
			Interfaces *interfaces,
			struct vuurmuur_config *cnf)
{
	int	result = 0;
	char	network[MAX_NET_ZONE] = "";


	/* safety */
	if(	rule_ptr == NULL || create == NULL || services == NULL ||
		zones == NULL || interfaces == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	/* if were on bash mode, alloc mem for the description */
	if(cnf->bash_out == TRUE)
	{
		if(!(create->description = malloc(sizeof(bash_description))))
		{
			(void)vrprint.error(-1, "Error", "malloc failed: %s (in: %s:%d).",
									strerror(errno), __FUNC__, __LINE__);
			return(-1);
		}
	}
	else
	{
		create->description = NULL;
	}

	/* first the protect rule */
	if(rule_ptr->action == AT_PROTECT)
	{
		if(debuglvl >= LOW)
			(void)vrprint.debug(__FUNC__, "action: %s, who: %s, danger: %s, source: %s",
								rules_itoaction(rule_ptr->action), rule_ptr->who,
								rule_ptr->danger, rule_ptr->source);

		/* description */
		if(cnf->bash_out && create->description != NULL)
		{
			snprintf(create->description, sizeof(bash_description), "rule: action: %s, who: %s, danger: %s, source: %s",
							rules_itoaction(rule_ptr->action), rule_ptr->who,
							rule_ptr->danger, rule_ptr->source);
		}

		/* get who */
		if(strcmp(rule_ptr->who, "") != 0)
		{
			if(rule_ptr->type == PROT_IPTABLES)
			{
				create->who_int = NULL;

				if(!(create->who = search_zonedata(debuglvl, zones, rule_ptr->who)))
				{
					(void)vrprint.error(-1, "Error", "zone '%s' not found (in: %s).", rule_ptr->who, __FUNC__);
					return(-1);
				}
			}
			else if(rule_ptr->type == PROT_PROC_INT)
			{
				create->who = NULL;
				create->who_int = NULL;

				if(!(create->who_int = search_interface(debuglvl, interfaces, rule_ptr->who)))
				{
					(void)vrprint.error(-1, "Error", "interface '%s' not found (in: %s).", rule_ptr->who, __FUNC__);
					return(-1);
				}
			}
			else
			{
				create->who = NULL;
				(void)vrprint.error(-1, "Error", "don't know what to do with '%s' for rule type '%d' (in: %s).", rule_ptr->who, rule_ptr->type, __FUNC__);
				return(-1);
			}
		}

		if(debuglvl >= MEDIUM)
			(void)vrprint.debug(__FUNC__, "calling get_danger_info() for danger...");

		result = get_danger_info(debuglvl, rule_ptr->danger, rule_ptr->source, &create->danger);
		if(result == 0)
		{
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "get_danger_info successfull.");
		}
		else
		{
			(void)vrprint.error(-1, "Error", "getting danger '%s' failed (in: %s).",
									rule_ptr->danger, __FUNC__);
			return(-1);
		}

		/* set the action */
		if(strlcpy(create->action, "protect", sizeof(create->action)) > sizeof(create->action))
		{
			(void)vrprint.error(-1, "Error", "buffer overflow (in: %s:%d).",
									__FUNC__, __LINE__);
			return(-1);
		}
	}
	/* network accept rule */
	else if(rule_ptr->type == PROT_IPTABLES &&
		(rule_ptr->action == AT_ACCEPT || rule_ptr->action == AT_QUEUE))
	{
		create->danger.solution = PROT_IPTABLES;

		/* description */
		if(cnf->bash_out && create->description != NULL)
		{
			snprintf(create->description, sizeof(bash_description), "rule: action: %s, service: %s",
								rules_itoaction(rule_ptr->action), rule_ptr->service);
		}

		/* get who */
		if(strcmp(rule_ptr->who, "") != 0)
		{
			if(rule_ptr->type == PROT_IPTABLES)
			{
				create->who_int = NULL;

				if(!(create->who = search_zonedata(debuglvl, zones, rule_ptr->who)))
				{
					(void)vrprint.error(-1, "Error", "zone '%s' not found (in: %s).", rule_ptr->who, __FUNC__);
					return(-1);
				}
			}
		}

		if(	strcasecmp(rule_ptr->service, "dhcp-client") == 0 ||
			strcasecmp(rule_ptr->service, "dhcp-server") == 0)
		{
			/* not much here */
			if(debuglvl >= MEDIUM)
				(void)vrprint.debug(__FUNC__, "network rule service '%s'", rule_ptr->service);
		}
		else
		{
			(void)vrprint.error(-1, "Error", "unknown service '%s' in network rule (in: %s:%d).",
									rule_ptr->service, __FUNC__, __LINE__);
			return(-1);
		}
	}
	/* separator */
	else if(rule_ptr->action == AT_SEPARATOR)
	{
		/* not much here */
		if(debuglvl >= MEDIUM)
			(void)vrprint.debug(__FUNC__, "rule is a separator");
	}
	/* normal rule */
	else
	{
		/* this is the rule */
		if(debuglvl >= LOW)
			(void)vrprint.debug(__FUNC__, "action: %s, service: %s, from: %s, to: %s",
								rules_itoaction(rule_ptr->action), rule_ptr->service,
								rule_ptr->from, rule_ptr->to);

		/* description */
		if(cnf->bash_out == TRUE && create->description != NULL)
		{
			snprintf(create->description, sizeof(bash_description), "rule: action: %s, service: %s, from: %s, to: %s",
								rules_itoaction(rule_ptr->action), rule_ptr->service,
								rule_ptr->from, rule_ptr->to);
		}

		/*
			GATHERING INFO FOR CREATING THE RULE
		*/
		create->active = rule_ptr->active;

		create->from = NULL;
		create->to = NULL;

		/*
			get 'from' if not firewall
		*/
		if(strncasecmp(rule_ptr->from, "firewall", 8) == 0)
		{
			/* we get the data later */
			create->from_firewall = TRUE;

			if(strcasecmp(rule_ptr->from, "firewall(any)") == 0)
				create->from_firewall_any = TRUE;
		}
		else if(strcasecmp(rule_ptr->from, "any") == 0)
		{
			/* we get the data later */
			create->from_any = TRUE;
		}
		else
		{
			/* get the pointer to the zonedata in the ZonedataList */
			if(!(create->from = search_zonedata(debuglvl, zones, rule_ptr->from)))
			{
				(void)vrprint.error(-1, "Error", "'from' zone '%s' not found (in: %s).",
								rule_ptr->from, __FUNC__);
				return(-1);
			}
		}

		/* normal network */
		if(strncasecmp(rule_ptr->to, "firewall", 8) == 0)
		{
			/* first check if we don't have two firewalls */
			if(create->from_firewall == TRUE)
			{
				(void)vrprint.error(-1, "Error", "'from' and 'to' are both set to firewall (%s service %s from %s to %s).",
								rules_itoaction(rule_ptr->action), rule_ptr->service,
								rule_ptr->from, rule_ptr->to);
				return(-1);
			}

			/* if from is any we cannot get the interfaces from it */
			if(create->from_any == FALSE)
			{
				/* assemble the network name */
				snprintf(network, sizeof(network), "%s.%s", create->from->network_name, create->from->zone_name);

				if(!(create->to = search_zonedata(debuglvl, zones, network)))
				{
					(void)vrprint.error(-1, "Error", "'to' zone '%s' not found (in: %s).",
								network, __FUNC__);
					return(-1);
				}
			}

			create->to_firewall = TRUE;

			if(strcasecmp(rule_ptr->to, "firewall(any)") == 0)
				create->to_firewall_any = TRUE;
		}
		else if(strcasecmp(rule_ptr->to, "any") == 0)
		{
			/* we get the data later */
			create->to_any = TRUE;
		}
		else
		{
			/* get the pointer to the zonedata in the ZonedataList */
			if(!(create->to = search_zonedata(debuglvl, zones, rule_ptr->to)))
			{
				(void)vrprint.error(-1, "Error", "'to' zone '%s' not found (in: %s).", rule_ptr->to, __FUNC__);
				return(-1);
			}
		}

		/* now get the data for the from-firewall */
		if(create->from_firewall == TRUE)
		{
			/* first check if we don't have two firewalls */
			if(create->to_firewall == TRUE)
			{
				(void)vrprint.error(-1, "Error", "'from' and 'to' are both set to firewall (%s service %s from %s to %s).", rules_itoaction(rule_ptr->action), rule_ptr->service, rule_ptr->from, rule_ptr->to);
				return(-1);
			}

			/* if to is any we cannot get the interfaces from it */
			if(create->to_any == FALSE)
			{
				/* get the pointer to the zonedata in the ZonedataList */
				snprintf(network, sizeof(network), "%s.%s", create->to->network_name, create->to->zone_name);

				if(!(create->from = search_zonedata(debuglvl, zones, network)))
				{
					(void)vrprint.error(-1, "Error", "'from' zone '%s' not found (in: %s).", network, __FUNC__);
					return(-1);
				}
			}
		}

		/* get the pointer to the services in the ServicesList */
		if(strcasecmp(rule_ptr->service, "any") == 0 || strcasecmp(rule_ptr->service, "all") == 0)
		{
			create->service_any = TRUE;
		}
		else
		{
			if(!(create->service = search_service(debuglvl, services, rule_ptr->service)))
			{
				(void)vrprint.error(-1, "Error", "service '%s' not found (in: %s).", rule_ptr->service, __FUNC__);
				return(-1);
			}
		}

		/* get the rule options */
		if(rule_ptr->opt != NULL)
			create->option = *rule_ptr->opt;

		/* determine which action to take (ACCEPT, DROP, REJECT etc.). */
		if(determine_action(debuglvl, rules_itoaction(rule_ptr->action), create->action, sizeof(create->action), &create->option) == 0)
		{
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "determine_action succes, create->action = %s",
												create->action);
		}
		else
		{
			(void)vrprint.error(-1, "Error", "could not determine action (in: %s).", __FUNC__);
			return(-1);
		}

		/* determine which chain to use. */
		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "calling determine_chain()...");

		if(determine_chain(debuglvl, rule_ptr, create->chain, sizeof(create->chain), &create->ruletype) == 0)
		{
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "determine_chain succesfull, create->chain = %s", create->chain);
		}
		else
		{
			(void)vrprint.error(-1, "Error", "could not determine chain (in: %s).", __FUNC__);
			return(-1);
		}

		/* QUEUE-ing can only be in input, output and forward rules */
		if(	rule_ptr->action == AT_QUEUE &&
			(create->ruletype != RT_INPUT && create->ruletype != RT_OUTPUT && create->ruletype != RT_FORWARD))
		{
			(void)vrprint.error(-1, "Error", "the QUEUE target can only be used in the input, output and forward chains (in: %s).", __FUNC__);
			return(-1);
		}

		if(	rule_ptr->action == AT_CHAIN &&
			(rule_ptr->opt == NULL || rule_ptr->opt->chain[0] == '\0'))
		{
			(void)vrprint.error(-1, "Error", "the CHAIN target needs option 'chain' to be set (in: %s:%d).",
												__FUNC__, __LINE__);
			return(-1);
		}

		/* make sure we only porfw to a host */
		if(create->ruletype == RT_PORTFW)
		{
			if(create->from_firewall == TRUE)
			{
				(void)vrprint.error(-1, "Error", "portforwarding is not allowed from the firewall (%s service %s from %s to %s).",
								rules_itoaction(rule_ptr->action), rule_ptr->service,
								rule_ptr->from, rule_ptr->to);
				return(-1);
			}

			if(create->to == NULL || create->to->type != TYPE_HOST)
			{
				(void)vrprint.error(-1, "Error", "portforwarding is only allowed to a host (%s service %s from %s to %s).",
								rules_itoaction(rule_ptr->action), rule_ptr->service,
								rule_ptr->from, rule_ptr->to);
				return(-1);
			}
		}
	
		/* make sure we dont redirect from the firewall */
		if(create->ruletype == RT_REDIRECT)
		{
			if(create->from_firewall == TRUE)
			{
				(void)vrprint.error(-1, "Error", "redirecting is not allowed from the firewall (%s service %s from %s to %s).",
								rules_itoaction(rule_ptr->action), rule_ptr->service,
								rule_ptr->from, rule_ptr->to);
				return(-1);
			}
		}

		/* don't snat to 'Any' */
		if(create->ruletype == RT_SNAT)
		{
			if(create->to_any == TRUE)
			{
				(void)vrprint.error(-1, "Error", "snat is not possible to 'Any' (%s service %s from %s to %s).",
								rules_itoaction(rule_ptr->action), rule_ptr->service,
								rule_ptr->from, rule_ptr->to);
				return(-1);
			}
		}

		/* don't masq to 'Any' */
		if(create->ruletype == RT_MASQ)
		{
			if(create->to_any == TRUE)
			{
				(void)vrprint.error(-1, "Error", "masq is not possible to 'Any' (%s service %s from %s to %s).",
								rules_itoaction(rule_ptr->action), rule_ptr->service,
								rule_ptr->from, rule_ptr->to);
				return(-1);
			}
		}

	} /* end else-protect */

	return(0);
}


/*	rules_init_list

	loads the rules from the backend

	Returncodes:
		 0: ok
		-1: error
*/
int
rules_init_list(const int debuglvl, Rules *rules, struct rgx_ *reg)
{
	FILE			*fp = NULL;
	int			retval = 0,
				count = 1,
				result = 0;
	char			line[MAX_RULE_LENGTH] = "";
	struct RuleData_	*rule_ptr = NULL;
	char			protect_warning_shown = FALSE;
	char			rule_name[32] = "";
	char			rules_found = FALSE;
	int			type = 0;


	/* safety */
	if(rules == NULL || reg == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	/* init */
	memset(rules, 0, sizeof(Rules));

	/*	setup the list: the cleanup function is set to NULL
		so it's the users responsibility to free memory. */
	if(d_list_setup(debuglvl, &rules->list, NULL) < 0)
	{
		(void)vrprint.error(-1, "Internal Error", "d_list_setup() failed (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	if(debuglvl >= MEDIUM)
		(void)vrprint.debug(__FUNC__, "rules_location: '%s'", conf.rules_location);

	/* open the rulesfile */
	if((fp = fopen(conf.rules_location, "r")))
	{
		rules->old_rulesfile_used = TRUE;

		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "opening rulesfile succeded.");

		/* run trough the file */
		while(fgets(line, sizeof(line), fp) != NULL)
		{
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "strlen(line) = %d", strlen(line));

			/* check if the line is a comment */
		//TODO what? what? what?
			if((strlen(line) <= 1) || (line[0] == '#'))
			{
				if(debuglvl >= HIGH)
					(void)vrprint.debug(__FUNC__, "skipping line because its a comment or its empty.");
			}
			/* no comment */
			else
			{
				/* alloc memory for the rule */
				if(!(rule_ptr = rule_malloc()))
				{
					(void)vrprint.error(-1, "Internal Error", "rule_malloc() failed: %s (in: %s:%d).",
											strerror(errno), __FUNC__, __LINE__);
					return(-1);
				}

				/* parse the line. We don't really care if it fails, we just ignore it. */
				if(rules_parse_line(debuglvl, line, rule_ptr, reg) < 0)
				{
					(void)vrprint.debug(__FUNC__, "parsing rule failed: %s", line);
				}
				else
				{
					/* protect rules are no longer supported in the main rules list */
					if(rule_ptr->action == AT_PROTECT)
					{
						if(protect_warning_shown == FALSE)
						{
							(void)vrprint.warning("Warning", "please note that the protect rules (e.g. anti-spoof) have been changed. Please recheck your networks and interfaces.");
							protect_warning_shown = TRUE;
						}

						free(rule_ptr);
						rule_ptr = NULL;
					}
					else
					{
						/* append to the rules list */
						if(!(d_list_append(debuglvl, &rules->list, rule_ptr)))
						{
							(void)vrprint.error(-1, "Internal Error", "d_list_append() failed (in: %s:%d).",
														__FUNC__, __LINE__);
							return(-1);
						}

						/* set the rule number */
						rule_ptr->number = count;
						count++;
					}
				}
			}
		}

		(void)vrprint.info("Info", "%d rules loaded.", count-1);

		if(fclose(fp) < 0)
		{
			(void)vrprint.error(-1, "Error", "closing rules file failed: %s (in: %s).", strerror(errno), __FUNC__);
			retval = -1;
		}
	}
	/* try to use the backend instead of the flat file */
	else
	{
		rules->old_rulesfile_used = FALSE;

		/* see if the rulesfile already exists in the backend */
		while(rf->list(debuglvl, rule_backend, rule_name, &type, CAT_RULES) != NULL)
		{
			if(debuglvl >= MEDIUM)
				(void)vrprint.debug(__FUNC__, "loading rules: '%s', type: %d",
										rule_name, type);

			if(strcmp(rule_name, "rules") == 0)
				rules_found = TRUE;
		}

		if(rules_found == FALSE)
		{
			if(rf->add(debuglvl, rule_backend, "rules", TYPE_RULE) < 0)
			{
				(void)vrprint.error(-1, "Internal Error", "rf->add() failed (in: %s:%d).",
										__FUNC__, __LINE__);
				return(-1);
			}
		}

		while((result = rf->ask(debuglvl, rule_backend, "rules", "RULE", line, sizeof(line), TYPE_RULE, 1)) == 1)
		{
			/* check if the line is a comment */
		//TODO what? what? what?
			if((strlen(line) <= 1) || (line[0] == '#'))
			{
				if(debuglvl >= HIGH)
					(void)vrprint.debug(__FUNC__, "skipping line because its a comment or its empty.");
			}
			/* no comment */
			else
			{
				/* alloc memory for the rule */
				if(!(rule_ptr = rule_malloc()))
				{
					(void)vrprint.error(-1, "Internal Error", "rule_malloc() failed: %s (in: %s:%d).",
											strerror(errno), __FUNC__, __LINE__);
					return(-1);
				}

				/* parse the line. We don't really care if it fails, we just ignore it. */
				if(rules_parse_line(debuglvl, line, rule_ptr, reg) < 0)
				{
					(void)vrprint.debug(__FUNC__, "parsing rule failed: %s", line);
				}
				else
				{
					/* protect rules are no longer supported in the main rules list */
					if(rule_ptr->action == AT_PROTECT)
					{
						if(protect_warning_shown == FALSE)
						{
							(void)vrprint.warning("Warning", "please note that the protect rules (e.g. anti-spoof) have been changed. Please recheck your networks and interfaces.");
							protect_warning_shown = TRUE;
						}

						free(rule_ptr);
						rule_ptr = NULL;
					}
					else
					{
						/* append to the rules list */
						if(!(d_list_append(debuglvl, &rules->list, rule_ptr)))
						{
							(void)vrprint.error(-1, "Internal Error", "d_list_append() failed (in: %s:%d).",
														__FUNC__, __LINE__);
							return(-1);
						}

						/* set the rule number */
						rule_ptr->number = count;
						count++;
					}
				}
			}
		}

		(void)vrprint.info("Info", "%d rules loaded.", count-1);
	}

	return(retval);
}


/*	rules_parse_line

	Returncodes:
		0: ok
		-1: error
*/
int
rules_parse_line(const int debuglvl, char *line, struct RuleData_ *rule_ptr, struct rgx_ *reg)
{
	int	line_pos = 0,	// position in line
		var_pos = 0;	// position in varible
	char	options[MAX_OPTIONS_LENGTH] = "";
	char	action_str[32] = "";


	/* safety first */
	if(line == NULL || rule_ptr == NULL || reg == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	/* decode the line */
	if(rules_decode_rule(debuglvl, line, MAX_RULE_LENGTH) < 0)
	{
		(void)vrprint.error(-1, "Internal Error", "decode rule failed (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	/* this should not happen, but it can't hurt to check, right? */
	if(strlen(line) > MAX_RULE_LENGTH)
	{
		(void)vrprint.error(-1, "Internal Error", "rule is too long (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}
	/* strip the newline */
	if(line[strlen(line)-1] == '\n')
		line[strlen(line)-1] = '\0';

	memset(options, 0, sizeof(options));

	if(debuglvl >= LOW)
		(void)vrprint.debug(__FUNC__, "rule: '%s'.", line);

	/* see if the rule is active */
	if(line[0] == ';')
	{
		if(debuglvl >= LOW)
			(void)vrprint.debug(__FUNC__, "rule is in-active.");

		rule_ptr->active = 0;

		line_pos = 1; 
	}
	else
	{
		if(debuglvl >= LOW)
			(void)vrprint.debug(__FUNC__, "rule is active.");

		rule_ptr->active = 1;
	}

	/* get the action */
	for(var_pos = 0; line_pos < sizeof(action_str)-1 && line[line_pos] != ' ' && line[line_pos] != '\0' && line[line_pos] != '\n'; line_pos++, var_pos++)
	{
		action_str[var_pos] = line[line_pos];
	}
	action_str[var_pos] = '\0';

	rule_ptr->action = rules_actiontoi(action_str);
	if(rule_ptr->action <= AT_ERROR || rule_ptr->action >= AT_TOO_BIG)
		return(-1);

	/*
		now we analyze the action
	*/
	if(rule_ptr->action == AT_PROTECT)
	{
		/*
			get the who, or 'against'
		*/
		for(line_pos++, var_pos = 0; var_pos < sizeof(RuleData.who) && line[line_pos] != ' ' && line[line_pos] != '\0' && line[line_pos] != '\n'; line_pos++, var_pos++)
		{
			rule_ptr->who[var_pos] = line[line_pos];
		}
		rule_ptr->who[var_pos] = '\0';

		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "protect: who: '%s'", rule_ptr->who);

		/*
			now check what kind of rule we have
		*/
		if(strcasecmp(rule_ptr->who, "against") == 0)
		{
			/*
				clear who, because we don't use it
			*/
			strcpy(rule_ptr->who, "");

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "rule is a PROT_PROC_SYS");

			/*
				okay, now lets see what kind of danger we are talking about
			*/
			for(line_pos++, var_pos = 0; var_pos < sizeof(RuleData.danger) && line[line_pos] != ' ' && line[line_pos] != '\0' && line[line_pos] != '\n'; line_pos++, var_pos++)
			{
				rule_ptr->danger[var_pos] = line[line_pos];
			}
			rule_ptr->danger[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "protect: danger: '%s'", rule_ptr->danger);
		}
		else
		{
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "now we know who (%s), let get the danger (but first check who).", rule_ptr->who);

			/*
				validate the who-zone
			*/
			if(validate_zonename(debuglvl, rule_ptr->who, 1, NULL, NULL, NULL, reg->zonename, VALNAME_VERBOSE) != 0)
			{
				(void)vrprint.error(-1, "Error", "invalid zonename: '%s' (in: %s).", rule_ptr->who, __FUNC__);
				return(-1);
			}

			/*
				get the keyword 'against'
			*/
			for(line_pos++, var_pos = 0; var_pos < strlen("against") && line[line_pos] != ' ' && line[line_pos] != '\0' && line[line_pos] != '\n'; line_pos++, var_pos++)
			{
				rule_ptr->danger[var_pos] = line[line_pos];
			}
			rule_ptr->danger[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "protect: keyword against: '%s'", rule_ptr->danger);

			/*
				if 'against' is missing, the rule is malformed, so we bail out screaming & kicking
			*/
			if(strcasecmp(rule_ptr->danger, "against") != 0)
			{
				(void)vrprint.error(-1, "Error", "bad rule syntax, keyword 'against' is missing: %s (in: %s).", line, __FUNC__);
				return(-1);
			}

			/*
				okay, now lets see what kind of danger we are talking about
			*/
			for(line_pos++, var_pos = 0; var_pos < sizeof(RuleData.danger) && line[line_pos] != ' ' && line[line_pos] != '\0' && line[line_pos] != '\n'; line_pos++, var_pos++)
			{
				rule_ptr->danger[var_pos] = line[line_pos];
			}
			rule_ptr->danger[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "protect: danger: '%s'", rule_ptr->danger);

			/*
				now determine if the danger is 'spoofing'
			*/
			if(strcasecmp(rule_ptr->danger, "spoofing") == 0)
			{
				/*
					get the 'from'
				*/
				for(line_pos++, var_pos = 0; var_pos < strlen("from") && line[line_pos] != ' ' && line[line_pos] != '\0' && line[line_pos] != '\n'; line_pos++, var_pos++)
				{
					rule_ptr->source[var_pos] = line[line_pos];
				}
				rule_ptr->source[var_pos] = '\0';

				if(debuglvl >= HIGH)
					(void)vrprint.debug(__FUNC__, "%s: protect: keyword from: '%s'", __FUNC__, rule_ptr->source);

				/*
					if 'from' is missing, the rule is malformed, so we bail out screaming & kicking
				*/
				if(strcasecmp(rule_ptr->source, "from") != 0)
				{
					(void)vrprint.error(-1, "Error", "bad rule syntax, keyword 'from' is missing: %s (in: %s).", line, __FUNC__);
					return(-1);
				}

				/*
					get the source
				*/
				for(line_pos++, var_pos = 0; var_pos < sizeof(RuleData.source) && line[line_pos] != ' ' && line[line_pos] != '\0' && line[line_pos] != '\n'; line_pos++, var_pos++)
				{
					rule_ptr->source[var_pos] = line[line_pos];
				}
				rule_ptr->source[var_pos] = '\0';

				if(debuglvl >= HIGH)
					(void)vrprint.debug(__FUNC__, "protect: source: '%s'", rule_ptr->source);
			}

			/*
				if don't use rule_ptr->source, clear it (just to be sure).
			*/
			else
			{
				strcpy(rule_ptr->source, "");
			}
		}
	}
	else
	{
		if(rule_ptr->action != AT_SEPARATOR)
		{
			/*
				first check for the keyword 'service'
			*/
			for(	line_pos++, var_pos = 0;
				var_pos < sizeof(RuleData.service) &&
					line[line_pos] != ' ' &&
					line[line_pos] != '\0' &&
					line[line_pos] != '\n' &&
					line_pos < strlen(line);
				line_pos++, var_pos++)
			{
				rule_ptr->service[var_pos] = line[line_pos];
			}
			rule_ptr->service[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "keyword service: '%s'.", rule_ptr->service);

			if(strcasecmp(rule_ptr->service, "service") != 0)
			{
				(void)vrprint.error(-1, "Error", "bad rule syntax, keyword 'service' is missing: %s (in: %s).", line, __FUNC__);
				return(-1);
			}

			/*
				get the service itself
			*/
			for(	line_pos++, var_pos = 0;
				var_pos < sizeof(RuleData.service) &&
					line[line_pos] != ' ' &&
					line[line_pos] != '\0' &&
					line[line_pos] != '\n' &&
					line_pos < strlen(line);
				line_pos++, var_pos++)
			{
				rule_ptr->service[var_pos] = line[line_pos];
			}
			rule_ptr->service[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "service: '%s'.", rule_ptr->service);

			if(strcmp(rule_ptr->service, "from") == 0)
			{
				(void)vrprint.error(-1, "Error", "bad rule syntax, keyword 'service' found, but has no value: %s (in: %s).", line, __FUNC__);
				return(-1);
			}

			/*
				validate the service name
			*/
			if(validate_servicename(debuglvl, rule_ptr->service, reg->servicename, VALNAME_QUIET) != 0)
			{
				(void)vrprint.error(-1, "Error", "invalid servicename: '%s' (in: %s:%d).", rule_ptr->service, __FUNC__, __LINE__);
				return(-1);
			}

			/*
				first check for the keyword 'from'
			*/
			for(	line_pos++, var_pos = 0;
				var_pos < sizeof(RuleData.from) &&
					line[line_pos] != ' ' &&
					line[line_pos] != '\0' &&
					line[line_pos] != '\n' &&
					line_pos < strlen(line);
				line_pos++, var_pos++)
			{
				rule_ptr->from[var_pos] = line[line_pos];
			}
			rule_ptr->from[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "keyword from: '%s'.", rule_ptr->from);

			if(strcasecmp(rule_ptr->from, "from") != 0)
			{
				(void)vrprint.error(-1, "Error", "bad rule syntax, keyword 'from' is missing: %s (in: %s).", line, __FUNC__);
				return(-1);
			}

			/*
				get the from itself
			*/
			for(	line_pos++, var_pos = 0;
				var_pos < sizeof(RuleData.from) &&
					line[line_pos] != ' ' &&
					line[line_pos] != '\0' &&
					line[line_pos] != '\n' &&
					line_pos < strlen(line);
				line_pos++, var_pos++)
			{
				rule_ptr->from[var_pos] = line[line_pos];
			}
			rule_ptr->from[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "from: '%s'.", rule_ptr->from);

			/*
				see if the from is actually the to keyword
			*/
			if(strcmp(rule_ptr->from, "to") == 0)
			{
				(void)vrprint.error(-1, "Error", "bad rule syntax, keyword 'from' found, but has no value: %s (in: %s).", line, __FUNC__);
				return(-1);
			}

			if(strncasecmp(rule_ptr->from, "firewall", 8) != 0)
			{
				/*
					now validate the from-zone
				*/
				if(validate_zonename(debuglvl, rule_ptr->from, 1, NULL, NULL, NULL, reg->zonename, VALNAME_VERBOSE) != 0)
				{
					(void)vrprint.error(-1, "Error", "invalid from-zonename: '%s' (in: %s).", rule_ptr->from, __FUNC__);
					return(-1);
				}
			}

			/*
				first check for the keyword 'to'
			*/
			for(	line_pos++, var_pos = 0;
				var_pos < sizeof(RuleData.to) &&
					line[line_pos] != ' ' &&
					line[line_pos] != '\0' &&
					line[line_pos] != '\n' &&
					line_pos < strlen(line);
				line_pos++, var_pos++)
			{
				rule_ptr->to[var_pos] = line[line_pos];
			}
			rule_ptr->to[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "keyword to: '%s'.", rule_ptr->to);

			if(strcasecmp(rule_ptr->to, "to") != 0)
			{
				(void)vrprint.error(-1, "Error", "bad rule syntax, keyword 'to' is missing: %s (in: %s).", line, __FUNC__);
				return(-1);
			}

			/*
				get to
			*/
			for(	line_pos++, var_pos = 0;
				var_pos < sizeof(RuleData.to) &&
					line[line_pos] != ' ' &&
					line[line_pos] != '\0' &&
					line[line_pos] != '\n' &&
					line_pos < strlen(line);
				line_pos++, var_pos++)
			{
				rule_ptr->to[var_pos] = line[line_pos];
			}
			rule_ptr->to[var_pos] = '\0';

			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "to: '%s'.", rule_ptr->to);

			/*
				see that to is not the keyword options
			*/
			if(strcmp(rule_ptr->to, "options") == 0)
			{
				(void)vrprint.error(-1, "Error", "bad rule syntax, keyword 'to' found, but has no value: %s (in: %s).", line, __FUNC__);
				return(-1);
			}

			if(strncasecmp(rule_ptr->to, "firewall", 8) != 0)
			{
				/*
					now validate the to-zone
				*/
				if(validate_zonename(debuglvl, rule_ptr->to, 1, NULL, NULL, NULL, reg->zonename, VALNAME_VERBOSE) != 0)
				{
					(void)vrprint.error(-1, "Error", "invalid zonename: '%s' (in: %s).", rule_ptr->to, __FUNC__);
					return(-1);
				}
			}
		}

		/*
			first check for the keyword 'options'
		*/
		for(	line_pos++, var_pos = 0;
			var_pos < sizeof(options) &&
				line[line_pos] != ' ' &&
				line[line_pos] != '\0' &&
				line[line_pos] != '\n' &&
				line_pos < strlen(line);
			line_pos++, var_pos++)
		{
			options[var_pos] = line[line_pos];
		}
		options[var_pos] = '\0';

		if(debuglvl >= MEDIUM)
			(void)vrprint.debug(__FUNC__, "keyword options: '%s'.", options);

		/*
			if this keyword exists we have options
		*/
		if(strcasecmp(options, "options") == 0)
		{
			/*
				get options: NOTE: whitespaces are allowed!
			*/
			for(	line_pos++, var_pos = 0;
				var_pos < sizeof(options) &&
					line[line_pos] != '\0' &&
					line[line_pos] != '\n' &&
					line_pos < MAX_RULE_LENGTH &&
					line_pos < strlen(line);
				line_pos++, var_pos++)
			{
				options[var_pos] = line[line_pos];
			}
			options[var_pos] = '\0';

			if(debuglvl >= LOW)
				(void)vrprint.debug(__FUNC__, "options: '%s'.", options);

			/* alloc options struct */
			if(!(rule_ptr->opt = ruleoption_malloc(debuglvl)))
			{
				(void)vrprint.error(-1, "Error", "malloc failed: %s (in: %s:%d).",
								strerror(errno), __FUNC__, __LINE__);
				return(-1);
			}

			/*
				now split them up
			*/
			if(rules_read_options(debuglvl, options, rule_ptr->opt) < 0)
			{
				(void)vrprint.error(-1, "Error", "parsing rule options failed for: '%s'.", line);

				free(rule_ptr->opt);
				rule_ptr->opt = NULL;

				return(-1);
			}
		}
		/*
			no options
		*/
		else
		{
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "rule has no options.");

			strcpy(options, "");

			rule_ptr->opt = NULL;
		}
	}

	return(0);
}


/*
	returns a pointer to the assembled line
*/
char *
rules_assemble_rule(const int debuglvl, struct RuleData_ *rule_ptr)
{
	char	*line = NULL,
		buf[512] = "",
		*option_ptr = NULL;
	size_t	bufsize = 0;

	/* safety */
	if(!rule_ptr)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(NULL);
	}

	/* assemble the line */
	if(	rule_ptr->action == AT_SEPARATOR)
	{
		snprintf(buf, sizeof(buf), "separator");
	}
	else
	{
		if(rule_ptr->active == TRUE)
		{
			snprintf(buf, sizeof(buf), "%s service %s from %s to %s",
								rules_itoaction(rule_ptr->action),
								rule_ptr->service,
								rule_ptr->from,
								rule_ptr->to);
		}
		else
		{
			snprintf(buf, sizeof(buf), ";%s service %s from %s to %s",
								rules_itoaction(rule_ptr->action),
								rule_ptr->service,
								rule_ptr->from,
								rule_ptr->to);
		}
	}

	if(!(option_ptr = rules_assemble_options_string(debuglvl, rule_ptr->opt, rules_itoaction(rule_ptr->action))))
		; /* no options, or error */
	else
	{
		strlcat(buf, " ", sizeof(buf));
		strlcat(buf, option_ptr, sizeof(buf));
		free(option_ptr);
	}

	strlcat(buf, "\n", sizeof(buf));
	/* assembling done */

	bufsize = strlen(buf) + 1; /* size of the line + nul */

	if(!(line = malloc(bufsize)))
	{
		(void)vrprint.error(-1, "Error", "malloc failed: %s (in: %s:%d).", strerror(errno), __FUNC__, __LINE__);
		return(NULL);
	}

	strlcpy(line, buf, bufsize);

	return(line);
}


/*
	TODO: mask! it should we only read/write for owner root, and nothing to the others
*/
static int
rules_write_file(const int debuglvl, Rules *rules, const char *rulesfile_location)
{
	FILE			*fp = NULL;
	int			retval = 0;
	d_list_node		*d_node = NULL;
	char			*line = NULL;
	struct RuleData_	*rule_ptr = NULL;

	/* safety */
	if(rulesfile_location == NULL || rules == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	/* open the rulesfile */
	if(!(fp = rules_file_open(rulesfile_location, "w+", 0)))
	{
		(void)vrprint.error(-1, "Error", "opening rulesfile '%s' failed: %s (in: %s).",
						rulesfile_location, strerror(errno), __FUNC__);
		return(-1);
	}

	if(debuglvl >= LOW)
		(void)vrprint.debug(__FUNC__, "number of rules %d.", rules->list.len);

	/* starting banner */
	fprintf(fp, "# Vuurmuur configfile, do not place comments in it, for they will be overwritten\n");

	/* loop trough the list */
	for(d_node = rules->list.top; d_node ; d_node = d_node->next)
	{
		if(!(rule_ptr = d_node->data))
		{
			(void)vrprint.error(-1, "Internal Error", "NULL pointer (in: %s:%d).", __FUNC__);

			(void)rules_file_close(fp, rulesfile_location);
			return(-1);
		}

		if(!(line = rules_assemble_rule(debuglvl, rule_ptr)))
		{
			(void)vrprint.error(-1, "Internal Error", "assembling rule failed (in: %s:%d).", __FUNC__);

			(void)rules_file_close(fp, rulesfile_location);
			return(-1);
		}

		/* now print the rule to the file */
		fprintf(fp, "%s", line);

		free(line);
		line = NULL;
	}

	/* print the end-of-file so we know all went fine */
	fprintf(fp, "# end of file\n");
	fflush(fp);

	/* close the rulesfile */
	retval = rules_file_close(fp, rulesfile_location);
	return(retval);
}


int
rules_save_list(const int debuglvl, Rules *rules, struct vuurmuur_config *cnf)
{
	int			result = 0;
	char			*line = NULL,
				eline[1024] = "";
	d_list_node		*d_node = NULL;
	struct RuleData_	*rule_ptr = NULL;
	char			overwrite = FALSE;


	/* safety */
	if(cnf == NULL || rules == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	if(rules->old_rulesfile_used == TRUE)
	{
		result = rules_write_file(debuglvl, rules, cnf->rules_location);
		if(result < 0)
			return(-1);
	}
	else
	{
		/* empty list, so clear all */
		if(rules->list.len == 0)
		{
			if(rf->tell(debuglvl, rule_backend, "rules", "RULE", "", 1, TYPE_RULE) < 0)
			{
				(void)vrprint.error(-1, "Internal Error", "rf->tell() failed (in: %s:%d).",
											__FUNC__, __LINE__);
				return(-1);
			}
		}
		else
		{
			overwrite = TRUE;

			/* loop trough the list */
			for(d_node = rules->list.top; d_node ; d_node = d_node->next)
			{
				if(!(rule_ptr = d_node->data))
				{
					(void)vrprint.error(-1, "Internal Error", "NULL pointer (in: %s:%d).",
											__FUNC__, __LINE__);
					return(-1);
				}

				if(!(line = rules_assemble_rule(debuglvl, rule_ptr)))
				{
					(void)vrprint.error(-1, "Internal Error", "rules_assemble_rule() failed (in: %s:%d).",
											__FUNC__, __LINE__);

					return(-1);
				}

				if(line[strlen(line)-1] == '\n')
					line[strlen(line)-1] = '\0';

				if(strlcpy(eline, line, sizeof(eline)) >= sizeof(eline))
				{
					(void)vrprint.error(-1, "Internal Error", "copy rule failed: buffer to small (in: %s:%d).",
											__FUNC__, __LINE__);
						return(-1);
				}

				free(line);
				line = NULL;


				/* encode */
				if(rules_encode_rule(debuglvl, eline, sizeof(eline)) < 0)
				{
					(void)vrprint.error(-1, "Internal Error", "encode rule failed (in: %s:%d).",
											__FUNC__, __LINE__);
						return(-1);
				}

				/* write to the backend */
				if(rf->tell(debuglvl, rule_backend, "rules", "RULE", eline, overwrite, TYPE_RULE) < 0)
				{
					(void)vrprint.error(-1, "Internal Error", "rf->tell() failed (in: %s:%d).",
												__FUNC__, __LINE__);
					return(-1);
				}

				overwrite = FALSE;
			}
		}
	}

	return(0);
}


/*	cleanup_ruleslist

	O(n) function: with n is the number of rules

	Returncodes:
		 0: ok
		-1: error
*/
int
rules_cleanup_list(const int debuglvl, Rules *rules)
{
	d_list_node		*d_node = NULL;
	struct RuleData_	*rule_ptr = NULL;


	/* safety */
	if(!rules)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s).", __FUNC__);
		return(-1);
	}

	/*
		loop trough the list to remove the options
	*/
	for(d_node = rules->list.top; d_node; d_node = d_node->next)
	{
		if(!(rule_ptr = d_node->data))
		{
			(void)vrprint.error(-1, "Internal Error", "NULL pointer (in: %s).", __FUNC__);
			return(-1);
		}

		/*	free the options. If there are none the
			'free_options' functions will take care
			of that.
		*/
		free_options(debuglvl, rule_ptr->opt);
		rule_ptr->opt = NULL;

		free(rule_ptr);
		rule_ptr = NULL;
	}

	/*
		cleanup the list
	*/
	if(d_list_cleanup(debuglvl, &rules->list) < 0)
		return(-1);

	return(0);
}


/*	rules_insert_list

	inserts a rule into the ruleslist at position 'place'.
	
	Returncodes:
		 0: ok
		-1: error
*/
int
rules_insert_list(const int debuglvl, Rules *rules, int place, struct RuleData_ *rule_ptr)
{
	struct RuleData_	*listrule_ptr = NULL;
	int			retval = 0;
	d_list_node		*d_node = NULL;


	/* safety */
	if(!rules || !rule_ptr)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).", __FUNC__, __LINE__);
		return(-1);
	}


	if(debuglvl >= HIGH)
		(void)vrprint.debug(__FUNC__, "insert at: %d. (list len is %d), number: %d, action: %s, service: %s, from: %s, to: %s, danger: %s, who: %s, source: %s.",
										place, 
										rules->list.len,
										rule_ptr->number,
										rules_itoaction(rule_ptr->action),
										rule_ptr->service,
										rule_ptr->from,
										rule_ptr->to,
										rule_ptr->danger,
										rule_ptr->who,
										rule_ptr->source);


	/* if we insert into empty list, we always insert at the top */
	if(rules->list.len == 0)
	{
		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "inserting into an empty list. Setting place to 1 (place was: %d).", place);

		place = 1;
	}

	/* handle inserting at the bottom of the list */
	if(place > rules->list.len)
	{
		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "place > rules_list->len (%d, %d). Setting place to %d.", place, rules->list.len, rules->list.len + 1);

		place = rules->list.len + 1;
	}


	/* handle insertion at the top of the list */
	if(place == 1)
	{
		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "place to insert: top");

		if(!(d_list_prepend(debuglvl, &rules->list, rule_ptr)))
		{
			(void)vrprint.error(-1, "Internal Error", "inserting the data to the top of list failed (in: %s:%d).", __FUNC__, __LINE__);
			return(-1);
		}

		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "d_list_prepend succes, now update numbers (place: %d)", place);

		rules_update_numbers(debuglvl, rules, place, 1);

		/* set number to 1 */
		rule_ptr->number = 1;

		/* we're done */
		return(0);
	}


	/*	now loop trough the list

		count: counts the number of rules we already processed
	*/
	for(d_node = rules->list.top; d_node ; d_node = d_node->next)
	{
		if(!(listrule_ptr = d_node->data))
		{
			(void)vrprint.error(-1, "Internal Error", "NULL pointer (in: %s:%d).", __FUNC__, __LINE__);
			return(-1);
		}
		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "entry: %s %s %s %s %s", rules_itoaction(listrule_ptr->action), listrule_ptr->service, listrule_ptr->danger, listrule_ptr->who, listrule_ptr->source);

		if(listrule_ptr->number == place - 1)
		{
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "place to insert: place: %d, %s %s %s %s %s", place, rules_itoaction(listrule_ptr->action), listrule_ptr->service, listrule_ptr->danger, listrule_ptr->who, listrule_ptr->source);

			if(!(d_list_insert_after(debuglvl, &rules->list, d_node, rule_ptr)))
			{
				(void)vrprint.error(-1, "Internal Error", "inserting the data into the list failed.");
				return(-1);
			}

			/* update numbers after count */
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "d_list_insert_after succes, now update numbers (place: %d)", place);

			rules_update_numbers(debuglvl, rules, place - 1, 1);

			/* set the number */
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "d_list_insert_after succes, now set rule_ptr->number to place: %d.", place);

			rule_ptr->number = place;

			/* we're done now */
			return(0);
		}
		/* we're not just there yet */
		else
		{
			if(debuglvl >= HIGH)
				(void)vrprint.debug(__FUNC__, "not the right place: %d, %s %s %s %s %s", place, rules_itoaction(listrule_ptr->action), listrule_ptr->service, listrule_ptr->danger, listrule_ptr->who, listrule_ptr->source);
		}
	}

	return(retval);
}


char *
rules_assemble_options_string(const int debuglvl, struct options *opt,
				const char *action)
{
	char	*option_ptr = NULL,
		options[MAX_OPTIONS_LENGTH] = "",
		*ports_ptr = NULL;
	char	redirect_port[6] = "",
		limit_string[6] = "",
		nfmark_string[9] = "";
	/* out_int="rtl8193" : out_int (7) = (1) " (1) " (1) \0 (1) = 11 */
	char	interfacestr[MAX_INTERFACE+11] = "";
	char	chainstr[48] = "";
	int	action_type = 0;

	/* safety - this is not an error! */
	if(opt == NULL || action == NULL)
		return(NULL);

	if(debuglvl >= LOW)
		(void)vrprint.debug(__FUNC__, "action: '%s'.", action);

	action_type = rules_actiontoi(action);
	if(action_type <= AT_ERROR || action_type >= AT_TOO_BIG)
	{
		(void)vrprint.error(-1, "Error", "unknown action '%s' "
				"(in: %s:%d).", action, __FUNC__, __LINE__);
		return(NULL);
	}

	/* init */
	strlcpy(options, "options ", sizeof(options));

	/* this one comes first so it's clearly visible in vuurmuur_conf */
	if(opt->in_int[0] != '\0')
	{
		snprintf(interfacestr, sizeof(interfacestr),
				"in_int=\"%s\",", opt->in_int);
		strlcat(options, interfacestr, sizeof(options));
	}
	/* this one comes first so it's clearly visible in vuurmuur_conf */
	if(opt->out_int[0] != '\0')
	{
		snprintf(interfacestr, sizeof(interfacestr),
				"out_int=\"%s\",", opt->out_int);
		strlcat(options, interfacestr, sizeof(options));
	}

	if(opt->chain[0] != '\0' && action_type == AT_CHAIN)
	{
		snprintf(chainstr, sizeof(chainstr),
				"chain=\"%s\",", opt->chain);
		strlcat(options, chainstr, sizeof(options));
	}

	/* the log options are also valid for the action LOG */
	if(opt->rule_log == TRUE || action_type == AT_LOG)
	{
		/* log option is only valid when action is not LOG */
		if(opt->rule_log == TRUE && action_type != AT_LOG)
			strlcat(options, "log,", sizeof(options));

		/* loglimit */
		if(opt->loglimit > 0)
		{
			snprintf(limit_string, sizeof(limit_string), "%d", opt->loglimit);

			strlcat(options, "loglimit=\"", sizeof(options));
			strlcat(options, limit_string, sizeof(options));
			strlcat(options, "\",", sizeof(options));
		}

		/* log prefix */
		if(opt->rule_logprefix == 1 && strcmp(opt->logprefix, "") != 0)
		{
			strlcat(options, "logprefix=\"", sizeof(options));
			strlcat(options, opt->logprefix, sizeof(options));
			strlcat(options, "\",", sizeof(options));
		}
	}

	/* queue, for portfw and redirect */
	if(opt->queue == 1 && (action_type == AT_PORTFW || action_type == AT_REDIRECT))
	{
		strlcat(options, "queue,", sizeof(options));
	}

	/* listenport and remoteport */
	if(action_type == AT_PORTFW)
	{
		if(opt->listenport == 1)
		{
			if(!(ports_ptr = list_to_portopts(debuglvl, &opt->ListenportList, "listenport")))
				;// no nothing?
			else
			{
				strlcat(options, ports_ptr, sizeof(options));
				free(ports_ptr);
				strlcat(options, ",", sizeof(options));
			}
		}
		if(opt->remoteport == 1)
		{
			if(!(ports_ptr = list_to_portopts(debuglvl, &opt->RemoteportList, "remoteport")))
				;// no nothing?
			else
			{
				strlcat(options, ports_ptr, sizeof(options));
				free(ports_ptr);
				strlcat(options, ",", sizeof(options));
			}
		}
	}

	if(opt->reject_option == 1)
	{
		if(action_type == AT_REJECT)
		{
			strlcat(options, "rejecttype=\"", sizeof(options));
			strlcat(options, opt->reject_type, sizeof(options));
			strlcat(options, "\",", sizeof(options));
		}
	}

	if(opt->redirectport > 0 && opt->redirectport <= 65535)
	{
		if(action_type == AT_REDIRECT)
		{
			snprintf(redirect_port, sizeof(redirect_port), "%d", opt->redirectport);

			strlcat(options, "redirectport=\"", sizeof(options));
			strlcat(options, redirect_port, sizeof(options));
			strlcat(options, "\",", sizeof(options));
		}
	}

	if(opt->markiptstate == 1)
	{
		if(action_type == AT_QUEUE)
		{
			strlcat(options, "markiptstate,", sizeof(options));
		}
	}

	if(opt->nfmark > 0)
	{
		snprintf(nfmark_string, sizeof(nfmark_string), "%lu", opt->nfmark);

		strlcat(options, "nfmark=\"", sizeof(options));
		strlcat(options, nfmark_string, sizeof(options));
		strlcat(options, "\",", sizeof(options));
	}

	/* limit */
	if(opt->limit > 0)
	{
		if(	action_type == AT_ACCEPT ||
			action_type == AT_QUEUE ||
			action_type == AT_PORTFW ||
			action_type == AT_REDIRECT ||
			action_type == AT_CHAIN ||
			action_type == AT_LOG)
		{
			snprintf(limit_string, sizeof(limit_string), "%u", opt->limit);

			strlcat(options, "limit=\"", sizeof(options));
			strlcat(options, limit_string, sizeof(options));
			strlcat(options, "\",", sizeof(options));

			if(opt->burst > 0)
			{
				snprintf(limit_string, sizeof(limit_string), "%u", opt->burst);

				strlcat(options, "burst=\"", sizeof(options));
				strlcat(options, limit_string, sizeof(options));
				strlcat(options, "\",", sizeof(options));
			}
		}
	}


	/* comment */
	if(opt->rule_comment == 1 && strcmp(opt->comment, "") != 0)
	{
		strlcat(options, "comment=\"", sizeof(options));
		strlcat(options, opt->comment, sizeof(options));
		strlcat(options, "\",", sizeof(options));
	}

	/* terminate the string */
	options[strlen(options)-1] = '\0';

	/* check if we did anything */
	if(strcmp(options, "options") != 0)
	{
		if(!(option_ptr = malloc(strlen(options)+1)))
		{
			(void)vrprint.error(-1, "Error", "malloc failed: %s.", strerror(errno));
			return(NULL);
		}
		else
		{
			strlcpy(option_ptr, options, strlen(options)+1);
			
			if(debuglvl >= MEDIUM)
				(void)vrprint.debug(__FUNC__, "'%s'.", option_ptr);
		}
	}

	if(debuglvl >= MEDIUM)
		(void)vrprint.debug(__FUNC__, "option_ptr: '%s'.", option_ptr);

	return(option_ptr);
}


/*
	Returncodes:
		-1: error
		 0: no change
		 1: change
*/
int
rules_compare_options(const int debuglvl, struct options *old_opt, struct options *new_opt, char *action)
{
	char	*old_str = NULL,
		*new_str = NULL;
	int	retval = 0;

	/* both NULL: no change */
	if(!old_opt && !new_opt)
	{
		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "options not changed (both NULL)");

		return(0);
	}

	/* if they are not the same: change */
	if((!old_opt && new_opt) || (old_opt && !new_opt))
	{
		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "options changed! (one NULL, other not)");

		return(1);
	}

	/* from here on, we are sure we have two options */
	if(!(old_str = rules_assemble_options_string(debuglvl, old_opt, action)))
		return(-1);

	if(!(new_str = rules_assemble_options_string(debuglvl, new_opt, action)))
	{
		free(old_str);
		return(-1);
	}

	if(strcmp(old_str, new_str) == 0)
		retval = 0;
	else
	{
		if(debuglvl >= HIGH)
			(void)vrprint.debug(__FUNC__, "options changed! (str compare)");

		retval = 1;
	}

	/* free the mem */
	free(old_str);
	free(new_str);

	return(retval);
}


/*
	TODO: compare active
*/
void *
search_rule(const int debuglvl, Rules *rules, struct RuleData_ *searchrule_ptr)
{
	d_list_node		*d_node = NULL;
	struct RuleData_	*listrule_ptr = NULL;

	/* safety */
	if(!rules || !searchrule_ptr)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).", __FUNC__, __LINE__);
		return(NULL);
	}

	for(d_node = rules->list.top; d_node ; d_node = d_node->next)
	{
		if(!(listrule_ptr = d_node->data))
		{
			(void)vrprint.error(-1, "Internal Error", "NULL pointer (in: %s:%d).", __FUNC__, __LINE__);
			return(NULL);
		}

		if(listrule_ptr->action == searchrule_ptr->action)
		{
			/* protect rule */
			if(searchrule_ptr->action == AT_PROTECT)
			{
				/* compare who */
				if(strcmp(listrule_ptr->who, searchrule_ptr->who) == 0)
				{
					/* compare source */
					if(strcmp(listrule_ptr->source, searchrule_ptr->source) == 0)
					{
						/* compare the danger */
						if(strcmp(listrule_ptr->danger, searchrule_ptr->danger) == 0)
						{
							return(listrule_ptr);
						}
					}
				}
			}

			/* normal rule */
			else
			{
				/* first compare the service-names */
				if(strcmp(listrule_ptr->service, searchrule_ptr->service) == 0)
				{
					/* comparing the 'from'-name */
					if(strcmp(listrule_ptr->from, searchrule_ptr->from) == 0)
					{
						/* comparing the 'to'-name */
						if(strcmp(listrule_ptr->to, searchrule_ptr->to) == 0)
						{
							/* comparing the rule options */
							if(rules_compare_options(debuglvl, listrule_ptr->opt, searchrule_ptr->opt, rules_itoaction(listrule_ptr->action)) == 0)
							{
								return(listrule_ptr);
							}
						}
					}
				}
			}
		}
	}

	return(NULL);
}


/* rules_read_options

    Call with the string with options and ouputs the option structure.

    TODO: this needs to be totally redesigned

*/
int
rules_read_options(const int debuglvl, char *optstr, struct options *op)
{
	int	retval = 0,
		x = 0,
		cur_pos = 0,
		o = 0,
		p = 0,
		trema = 0;
	char	curopt[512] = "",
		portstring[512] = "";

	/* safety */
	if(optstr == NULL || op == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).", __FUNC__, __LINE__);
		return(-1);
	}

	if(debuglvl >= HIGH)
		(void)vrprint.debug(__FUNC__, "options: '%s', strlen(optstr): %d", optstr, strlen(optstr));

	/* check if we even got a string to disassemble */
	if(strlen(optstr) == 0)
	{
		if(debuglvl >= MEDIUM)
			(void)vrprint.debug(__FUNC__, "no options.");

		return(0);
	}


	while((strlen(optstr) >= x))
	{
		curopt[cur_pos] = optstr[x];
		cur_pos++;

		/* between the trema's (") don't use the comma as a separator. */
		if((optstr[x] == '"') && (trema == 1))
		{
			trema = 2;
		}
		if((optstr[x] == '"') && (trema == 0))
		{
			trema = 1;
		}

		if(((optstr[x] == ',') && ((trema == 0) || (trema == 2))) || (optstr[x] == '\0'))
		{
			curopt[cur_pos - 1] = '\0';
			cur_pos = 0;
		}
		x++;

		/* reset trema, so we can have more trema pairs. */
		if(trema == 2)
			trema = 0;

		/* we are done */
		if(cur_pos == 0)
		{
			if(debuglvl >= LOW)
				(void)vrprint.debug(__FUNC__, "curopt: '%s'.", curopt);

			/* error message for a missing trema */
			if(trema == 1)
			{
				(void)vrprint.error(-1, "Error", "unbalanced \" in rule (in: %s:%d).", __FUNC__, __LINE__);
				return(-1);
			}


			/*
				start parsing the options
			*/

			/* log - log the rule? */
			if(strcmp(curopt, "log") == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "logging enabled.");

				op->rule_log = 1;
			}
			/* loglimit */
			else if(strncmp(curopt, "loglimit", strlen("loglimit")) == 0)
			{
				for(	p = 0, o = strlen("loglimit") + 1;
					o < strlen(curopt) && p < sizeof(portstring);
					o++)
				{
					if(curopt[o] != '\"')
					{
						portstring[p] = curopt[o];
						p++;
					}
				}
				portstring[p] = '\0';

				op->loglimit = atoi(portstring);
				op->logburst = op->loglimit * 2;

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "loglimit: %d, logburst %d.", op->loglimit, op->logburst);
			}
			/* limit */
			else if(strncmp(curopt, "limit", strlen("limit")) == 0)
			{
				for(	p = 0, o = strlen("limit") + 1;
					o < strlen(curopt) && p < sizeof(portstring);
					o++)
				{
					if(curopt[o] != '\"')
					{
						portstring[p] = curopt[o];
						p++;
					}
				}
				portstring[p] = '\0';

				op->limit = atoi(portstring);

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "limit: %d.", op->limit);
			}
			/* burst */
			else if(strncmp(curopt, "burst", strlen("burst")) == 0)
			{
				for(	p = 0, o = strlen("burst") + 1;
					o < strlen(curopt) && p < sizeof(portstring);
					o++)
				{
					if(curopt[o] != '\"')
					{
						portstring[p] = curopt[o];
						p++;
					}
				}
				portstring[p] = '\0';

				op->burst = atoi(portstring);

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "burst: %d.", op->burst);
			}
			/* mark the iptablesstate? */
			else if(strcmp(curopt, "markiptstate") == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "marking iptstate enabled.");

				op->markiptstate = 1;
			}
			/* queue instead of accept (portfw and redirect) */
			else if(strcmp(curopt, "queue") == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "queue'ing enabled.");

				op->queue = 1;
			}
			/* int */
			else if(strncmp(curopt, "int", 3) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "int (old for in_int) option.");

				for(	p = 0, o = strlen("int") + 2;
					p < sizeof(op->in_int) && o < strlen(curopt) - 1;
					o++, p++)
				{
					op->in_int[p] = curopt[o];
				}
				op->in_int[p] = '\0';

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "in_int: '%s'.", op->in_int);
			}
			/* in_int */
			else if(strncmp(curopt, "in_int", 6) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "in_int option.");

				for(	p = 0, o = strlen("in_int") + 2;
					p < sizeof(op->in_int) && o < strlen(curopt) - 1;
					o++, p++)
				{
					op->in_int[p] = curopt[o];
				}
				op->in_int[p] = '\0';

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "in_int: '%s'.", op->in_int);
			}
			/* in_int */
			else if(strncmp(curopt, "out_int", 7) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "out_int option.");

				for(	p = 0, o = strlen("out_int") + 2;
					p < sizeof(op->out_int) && o < strlen(curopt) - 1;
					o++, p++)
				{
					op->out_int[p] = curopt[o];
				}
				op->out_int[p] = '\0';

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "out_int: '%s'.", op->out_int);
			}
			/* remoteport - for portforwarding */
			else if(strncmp(curopt, "remoteport", strlen("remoteport")) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "remoteport specified.");

				/* copy the string containing the ports */
				for(	p = 0, o = strlen("remoteport") + 1;
					o <= strlen(curopt) && p < sizeof(portstring);
					o++, p++)
				{
					portstring[p] = curopt[o];
				}
//TODO: no NULL?

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "remoteport string: '%s'.", portstring);

				if(portopts_to_list(debuglvl, portstring, &op->RemoteportList) < 0)
				{
					(void)vrprint.error(-1, "Error", "parsing remoteport option failed. Please check the syntax of the rule.");
					return(-1);
				}

				op->remoteport = 1;
			}
			/* listenport - for portforwarding */
			else if(strncmp(curopt, "listenport", strlen("listenport")) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "listenport specified.");

				/* copy the string containing the ports */
				for(	p = 0, o = strlen("listenport") + 1;
					o <= strlen(curopt) && p < sizeof(portstring);
					o++, p++)
				{
					portstring[p] = curopt[o];
				}
//TODO: no NULL?
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "listenport string: '%s'.", portstring);

				if(portopts_to_list(debuglvl, portstring, &op->ListenportList) < 0)
				{
					(void)vrprint.error(-1, "Error", "parsing listenport option failed. Please check the syntax of the rule.");
					return(-1);
				}

				op->listenport = 1;
			}
			/* rule comment */
			else if(strncmp(curopt, "comment", strlen("comment")) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "comment.");

				for(	p = 0, o = strlen("comment") + 2;
					o < strlen(curopt) - 1 && p < sizeof(op->comment);
					o++, p++)
				{
					op->comment[p] = curopt[o];
				}
				op->comment[p] = '\0';
				op->rule_comment = 1;
			}
			/* logprefix, max 29 characters long. */
			else if(strncmp(curopt, "logprefix", strlen("logprefix")) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "logprefix.");

				for(	p = 0, o = strlen("logprefix") + 2;
					p < 12 && o < strlen(curopt) - 1 && p < sizeof(op->logprefix);
					o++, p++)
				{
					op->logprefix[p] = curopt[o];
				}
				op->logprefix[p] = '\0';

				if(strlen(op->logprefix) > 14)
				{
//TODO: not disable, but truncate */
					(void)vrprint.warning("Warning", "logprefix is too long. Maximum length is 14 characters.");
					op->rule_logprefix = 0;
					op->logprefix[0] = '\0';
				}
				else
				{
					op->rule_logprefix=1;
				}
			}
			/* redirectport */
			else if(strncmp(curopt, "redirectport", strlen("redirectport")) == 0)
			{
				for(	p = 0, o = strlen("redirectport") + 1;
					o < strlen(curopt) && p < sizeof(portstring);
					o++)
				{
					if(curopt[o] != '\"')
					{
						portstring[p] = curopt[o];
						p++;
					}
				}
				portstring[p] = '\0';

				op->redirectport = atoi(portstring);
				if(op->redirectport <= 0 || op->redirectport > 65535)
				{
					(void)vrprint.error(-1, "Error", "redirectport must be 1-65535.");
					op->redirectport = 0;
					return(-1);
				}

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "redirectport: %d, %s", op->redirectport, portstring);
			}
			/* redirectport */
			else if(strncmp(curopt, "nfmark", strlen("nfmark")) == 0)
			{
				for(	p = 0, o = strlen("nfmark") + 1;
					o < strlen(curopt) && p < sizeof(portstring);
					o++)
				{
					if(curopt[o] != '\"')
					{
						portstring[p] = curopt[o];
						p++;
					}
				}
				portstring[p] = '\0';

				op->nfmark = strtoul(portstring, (char **)NULL, 10);

				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "nfmark: %lu, %s", op->nfmark, portstring);
			}
			/* reject type */
			else if(strncmp(curopt, "rejecttype", strlen("rejecttype")) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "rejecttype.");

				for(	p = 0, o = strlen("rejecttype") + 1;
					o < strlen(curopt) && o < 23 + strlen("rejecttype") + 1 && p < sizeof(op->reject_type);
					o++)
				{ /* 23 is from the length of the string */

					if(curopt[o] != '\"')
					{
						op->reject_type[p] = curopt[o];
						p++;
					}
				}
				op->reject_type[p] = '\0';
				op->reject_option = 1;

				/* check if the option is valid. */
				if(	strcmp(op->reject_type, "icmp-net-unreachable") == 0);
				else if(strcmp(op->reject_type, "icmp-host-unreachable") == 0);
				else if(strcmp(op->reject_type, "icmp-proto-unreachable") == 0);
				else if(strcmp(op->reject_type, "icmp-port-unreachable") == 0);
				else if(strcmp(op->reject_type, "icmp-net-prohibited") == 0);
				else if(strcmp(op->reject_type, "icmp-host-prohibited") == 0);
				else if(strcmp(op->reject_type, "tcp-reset") == 0);
				else
				{
					(void)vrprint.error(-1, "Error", "%s is not a valid reject-type.", op->reject_type);

					op->reject_option = 0;
					return(-1);
				}
			}
			/* chain */
			else if(strncmp(curopt, "chain", strlen("chain")) == 0)
			{
				if(debuglvl >= MEDIUM)
					(void)vrprint.debug(__FUNC__, "chain.");

				for(	p = 0, o = strlen("chain") + 2;
					o < strlen(curopt) - 1 && p < sizeof(op->chain);
					o++, p++)
				{
					op->chain[p] = curopt[o];
				}
				op->chain[p] = '\0';
			}
			/* unknown option */
			else
			{
				(void)vrprint.warning("Warning", "unknown rule option '%s'.", curopt);
				//return(-1);
			}
		}
	}

	if(debuglvl >= MEDIUM)
		(void)vrprint.debug(__FUNC__, "** end **, return = %d", retval);

	return(retval);
}


/*
	AT_ERROR = -1,
	AT_ACCEPT,
	AT_DROP,
	AT_REJECT,
	AT_LOG,
	AT_PORTFW,
	AT_REDIRECT,
	AT_SNAT,
	AT_MASQ,
	AT_QUEUE,
	AT_CHAIN,

	AT_SEPARATOR,
*/

char *actions[] =
{
	"Accept",
	"Drop",
	"Reject",
	"Log",
	"Portfw",
	"Redirect",
	"Snat",
	"Masq",
	"Queue",
	"Chain",
	"Protect",
	"Separator",
	"ERROR",
};


char *
rules_itoaction(int i)
{
	/* i needs to be smaller than the number of items in the array */
	if(i >= ( sizeof(actions) / sizeof(actions[0]) ) -1 )
		return(NULL);

	return(actions[i]);
}


char *actions_cap[] =
{
	"ACCEPT",
	"DROP",
	"REJECT",
	"LOG",
	"PORTFW",
	"REDIRECT",
	"SNAT",
	"MASQ",
	"QUEUE",
	"CHAIN",
	"PROTECT",
	"SEPARATOR",
	"ERROR",
};


char *
rules_itoaction_cap(int i)
{
	/* status needs to be smaller than the number of items in the array */
	if(i >= ( sizeof(actions_cap) / sizeof(actions_cap[0]) ) -1 )
		return(NULL);

	return(actions_cap[i]);
}


/*	rules_actiontoi

	Converts the action into an int as defined by enum action_types.
*/
int
rules_actiontoi(const char *action)
{
	if(action == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(AT_ERROR);
	}

	if(strcasecmp(action, "accept") == 0)
		return(AT_ACCEPT);
	else if(strcasecmp(action, "drop") == 0)
		return(AT_DROP);
	else if(strcasecmp(action, "reject") == 0)
		return(AT_REJECT);
	else if(strcasecmp(action, "log") == 0)
		return(AT_LOG);
	else if(strcasecmp(action, "portfw") == 0)
		return(AT_PORTFW);
	else if(strcasecmp(action, "redirect") == 0)
		return(AT_REDIRECT);
	else if(strcasecmp(action, "snat") == 0)
		return(AT_SNAT);
	else if(strcasecmp(action, "masq") == 0)
		return(AT_MASQ);
	else if(strcasecmp(action, "queue") == 0)
		return(AT_QUEUE);
	else if(strcasecmp(action, "chain") == 0)
		return(AT_CHAIN);
	else if(strcasecmp(action, "sepparator") == 0 ||
		strcasecmp(action, "separator") == 0)
		return(AT_SEPARATOR);
	else if(strcasecmp(action, "protect") == 0)
		return(AT_PROTECT);
	else
	{
		(void)vrprint.error(-1, "Error", "unknown action '%s' (in: %s:%d).",
							action, __FUNC__, __LINE__);
		return(AT_ERROR);
	}
}


struct RuleData_ *
rules_create_protect_rule(const int debuglvl, char *action, /*@null@*/ char *who, char *danger, /*@null@*/ char *source)
{
	struct RuleData_	*rule_ptr = NULL;

	/* safety */
	if(!danger || !action)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s).", __FUNC__);
		return(NULL);
	}

	/* get mem (rule_malloc will report error) */
	if(!(rule_ptr = rule_malloc()))
		return(NULL);

	rule_ptr->action = rules_actiontoi(action);
	if(rule_ptr->action <= AT_ERROR || rule_ptr->action >= AT_TOO_BIG)
		return(NULL);

	if(rule_ptr->action == AT_ACCEPT)
	{
		/* who do we protect */
		strlcpy(rule_ptr->service, danger, sizeof(rule_ptr->service));
	}
	else
	{
		/* who do we protect */
		strlcpy(rule_ptr->who, who, sizeof(rule_ptr->who));

		/* and against what? */
		strlcpy(rule_ptr->danger, danger, sizeof(rule_ptr->danger));

		if(source != NULL)
		{
			/* from which source */
			strlcpy(rule_ptr->source, source, sizeof(rule_ptr->source));
		}
	}

	return(rule_ptr);
}


/*	rules_chain_in_list

	returns:
		 1: if chain is in the list
		 0: if not
		-1: error
*/
int
rules_chain_in_list(const int debuglvl, d_list *list, char *chainname)
{
	char		*str = NULL;
	d_list_node	*d_node = NULL;

	/* safety */
	if(list == NULL || chainname == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	for(d_node = list->top; d_node; d_node = d_node->next)
	{
		if(!(str = d_node->data))
		{
			(void)vrprint.error(-1, "Internal Error", "NULL pointer (in: %s:%d).",
									__FUNC__, __LINE__);
			return(-1);
		}

		if(strcmp(str, chainname) == 0)
		{
			return(1);
		}
	}

	return(0);
}


/*	rules_get_custom_chains

	Looks at all rules a puts all custom chain names
	as strings in a list.

	returncodes:
		 0: ok
		-1: error
*/
int
rules_get_custom_chains(const int debuglvl, Rules *rules)
{
	struct RuleData_	*rule_ptr = NULL;
	d_list_node		*d_node = NULL;
	char			*str = NULL;
	size_t			size = 0;

	/* safety */
	if(rules == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	if(d_list_setup(debuglvl, &rules->custom_chain_list, free) < 0)
	{
		(void)vrprint.error(-1, "Internal Error", "d_list_setup() failed (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	for(d_node = rules->list.top; d_node; d_node = d_node->next)
	{
		if(!(rule_ptr = d_node->data))
		{
			(void)vrprint.error(-1, "Internal Error", "NULL pointer (in: %s:%d).",
									__FUNC__, __LINE__);
			return(-1);
		}

		if(rule_ptr->opt != NULL)
		{
			if(rule_ptr->opt->chain[0] != '\0')
			{
				/* see if the chain is already in our list */
				if(rules_chain_in_list(debuglvl, &rules->custom_chain_list, rule_ptr->opt->chain) == 0)
				{
					size = strlen(rule_ptr->opt->chain) + 1;

					str = malloc(size);
					if(str == NULL)
					{
						(void)vrprint.error(-1, "Error", "malloc failed: %s (in: %s:%d).",
										strerror(errno), __FUNC__, __LINE__);
						return(-1);
					}

					strlcpy(str, rule_ptr->opt->chain, size);

					if(d_list_append(debuglvl, &rules->custom_chain_list, str) < 0)
					{
						(void)vrprint.error(-1, "Internal Error", "d_list_append() failed (in: %s:%d).",
												__FUNC__, __LINE__);
						return(-1);
					}
				}
			}
		}
	}

	return(0);
}


/*	rules_get_system_chains

	Gets all chain currently in the filter table on the system.

	Returncodes:
		-1: error
		 0: ok
*/
int
rules_get_system_chains(const int debuglvl, Rules *rules, struct vuurmuur_config *cnf)
{
	char		line[128] = "",
			cmd[128] = "";
	FILE		*p = NULL;
	char		chainname[32] = "";
	char		*name = NULL;
	size_t		size = 0;

	/* safety */
	if(cnf == NULL || rules == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	if(d_list_setup(debuglvl, &rules->system_chain_list, free) < 0)
	{
		(void)vrprint.error(-1, "Internal Error", "d_list_setup() failed (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	if(cnf->iptables_location[0] == '\0')
	{
		(void)vrprint.error(-1, "Internal Error", "cnf->iptables_location is empty (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	/* commandline */
	snprintf(cmd, sizeof(cmd), "%s -t filter -nL", cnf->iptables_location);

	/* open the pipe to the command */
	if((p = popen(cmd, "r")))
	{
		/* loop through the result */
		while(fgets(line, sizeof(line), p) != NULL)
		{
			if(strncmp("Chain", line, 5) == 0)
			{
				sscanf(line, "Chain %32s", chainname);

				size = strlen(chainname) + 1;

				name = malloc(size);
				if(name == NULL)
				{
					(void)vrprint.error(-1, "Error", "malloc failed: %s (in: %s:%d).",
									strerror(errno), __FUNC__, __LINE__);
					return(-1);
				}

				strlcpy(name, chainname, size);

				if(d_list_append(debuglvl, &rules->system_chain_list, name) < 0)
				{
					(void)vrprint.error(-1, "Internal Error", "d_list_append() failed (in: %s:%d).",
											__FUNC__, __LINE__);
					return(-1);
				}
			}
		}

		/* finally close the pipe */
		pclose(p);
	}
	else
	{
		if(debuglvl >= MEDIUM)
			(void)vrprint.debug(__FUNC__, "popen() failed");
	}

	return(0);
}


/*	rules_encode_rule

	convert all " to \"

	Return
		-1: error
		 0: ok
*/
int
rules_encode_rule(const int debuglvl, char *rulestr, size_t size)
{
	char	line[1024] = "";
	int	i = 0,
		x = 0;

	/* safety */
	if(rulestr == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	for(i = 0, x = 0; i < strlen(rulestr) && x < size; i++, x++)
	{
		if(	rulestr[i] == '\"' &&
			(i == 0 || rulestr[i-1] != '\\')) /* check for not double encoding */
		{
			line[x] = '\\';
			x++;
		}

		line[x] = rulestr[i];
	}
	line[x] = '\0';

	if(strlcpy(rulestr, line, size) >= size)
	{
		(void)vrprint.error(-1, "Internal Error", "encoding rule failed: buffer to small (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	return(0);
}

int
rules_decode_rule(const int debuglvl, char *rulestr, size_t size)
{
	char	line[1024] = "";
	int	i = 0,
		x = 0;
	size_t	len = 0;

	/* safety */
	if(rulestr == NULL)
	{
		(void)vrprint.error(-1, "Internal Error", "parameter problem (in: %s:%d).",
									__FUNC__, __LINE__);
		return(-1);
	}

	for(i = 0, x = 0; i < strlen(rulestr) && x < size; i++)
	{
		if(rulestr[i] == '\\' && rulestr[i+1] == '\"')
		{
			/* nothing */
		}
		else
		{
			line[x] = rulestr[i];
			x++;
		}
	}
	line[x] = '\0';

	/* this error should be impossible... thats why we check ;-) */
	len = strlcpy(rulestr, line, size);
	if(len >= size)
	{
		(void)vrprint.error(-1, "Internal Error", "decoding rule failed: buffer to small: %u>=%u (in: %s:%d).",
									len, size, __FUNC__, __LINE__);
		return(-1);
	}

	return(0);
}

